<?php
/**
 * API: التحقق من الترخيص
 * Endpoint: /api/verify-license
 */

define('API_ACCESS', true);
require_once 'config.php';

// التحقق من الطلب والتوقيع الرقمي
verify_request(array('license_key', 'site_url', 'product'));

$license_key = sanitize_text_field($_POST['license_key']);
$site_url = sanitize_text_field($_POST['site_url']);
$product = sanitize_text_field($_POST['product']);
$version = sanitize_text_field($_POST['version'] ?? '1.0.0');

// تنظيف URL
$site_url = rtrim($site_url, '/');
$site_url = str_replace(array('http://', 'https://'), '', $site_url);

try {
    // البحث عن الترخيص
    $stmt = $pdo->prepare("
        SELECT l.*, a.id as activation_id
        FROM licenses l
        LEFT JOIN activations a ON l.id = a.license_id AND a.site_url = :site_url
        WHERE l.license_key = :license_key 
        AND l.product = :product
    ");

    $stmt->execute(array(
        ':license_key' => $license_key,
        ':site_url' => $site_url,
        ':product' => $product
    ));

    $license = $stmt->fetch();

    if (!$license) {
        send_json_response(array(
            'success' => false,
            'status' => 'invalid',
            'message' => 'Invalid license key'
        ), 404);
    }

    // التحقق من الحالة
    $now = new DateTime();
    $expires_at = $license['expires_at'] ? new DateTime($license['expires_at']) : null;

    $status = $license['status'];

    // التحقق من انتهاء الصلاحية
    if ($expires_at && $now > $expires_at) {
        $status = 'expired';

        // تحديث الحالة في قاعدة البيانات
        $pdo->prepare("UPDATE licenses SET status = 'expired' WHERE id = :id")
            ->execute(array(':id' => $license['id']));
    }

    // تحديث آخر فحص
    $pdo->prepare("UPDATE licenses SET last_check = NOW() WHERE id = :id")
        ->execute(array(':id' => $license['id']));

    if ($license['activation_id']) {
        $pdo->prepare("UPDATE activations SET last_check = NOW(), version = :version WHERE id = :id")
            ->execute(array(
                ':id' => $license['activation_id'],
                ':version' => $version
            ));
    }

    // تسجيل الحدث
    $pdo->prepare("
        INSERT INTO license_logs (license_id, action, details, ip_address)
        VALUES (:license_id, 'verify', :details, :ip)
    ")->execute(array(
                ':license_id' => $license['id'],
                ':details' => json_encode(array('site_url' => $site_url, 'version' => $version)),
                ':ip' => $_SERVER['REMOTE_ADDR']
            ));

    // الاستجابة
    send_json_response(array(
        'success' => true,
        'status' => $status,
        'data' => array(
            'email' => $license['email'],
            'plan' => $license['plan'],
            'expires_at' => $license['expires_at'],
            'site_url' => $site_url,
            'max_sites' => $license['max_sites']
        )
    ));

} catch (Exception $e) {
    send_json_response(array(
        'success' => false,
        'message' => 'Server error: ' . $e->getMessage()
    ), 500);
}

function sanitize_text_field($str)
{
    return htmlspecialchars(strip_tags(trim($str)), ENT_QUOTES, 'UTF-8');
}
