<?php
/**
 * API: تفعيل الترخيص
 * Endpoint: /api/activate-license
 */

define('API_ACCESS', true);
require_once 'config.php';

// التحقق من الطلب
verify_request(array('license_key', 'site_url', 'admin_email', 'product'));

$license_key = sanitize_text_field($_POST['license_key']);
$site_url = sanitize_text_field($_POST['site_url']);
$admin_email = sanitize_text_field($_POST['admin_email']);
$product = sanitize_text_field($_POST['product']);
$site_name = sanitize_text_field($_POST['site_name'] ?? '');

// تنظيف URL
$site_url = rtrim($site_url, '/');
$site_url = str_replace(array('http://', 'https://'), '', $site_url);

try {
    // البحث عن الترخيص
    $stmt = $pdo->prepare("
        SELECT * FROM licenses 
        WHERE license_key = :license_key 
        AND product = :product
    ");

    $stmt->execute(array(
        ':license_key' => $license_key,
        ':product' => $product
    ));

    $license = $stmt->fetch();

    if (!$license) {
        send_json_response(array(
            'success' => false,
            'message' => 'مفتاح الترخيص غير صحيح'
        ), 404);
    }

    // التحقق من الحالة
    if ($license['status'] === 'expired') {
        send_json_response(array(
            'success' => false,
            'message' => 'هذا الترخيص منتهي الصلاحية'
        ), 403);
    }

    if ($license['status'] === 'suspended') {
        send_json_response(array(
            'success' => false,
            'message' => 'هذا الترخيص معلّق. يرجى التواصل مع الدعم'
        ), 403);
    }

    // التحقق من عدد المواقع المفعّلة
    $stmt = $pdo->prepare("
        SELECT COUNT(*) as count 
        FROM activations 
        WHERE license_id = :license_id
    ");

    $stmt->execute(array(':license_id' => $license['id']));
    $activation_count = $stmt->fetchColumn();

    // التحقق من وجود تفعيل سابق لنفس الموقع
    $stmt = $pdo->prepare("
        SELECT id FROM activations 
        WHERE license_id = :license_id 
        AND site_url = :site_url
    ");

    $stmt->execute(array(
        ':license_id' => $license['id'],
        ':site_url' => $site_url
    ));

    $existing_activation = $stmt->fetch();

    if ($existing_activation) {
        // الموقع مفعّل مسبقاً - تحديث البيانات فقط
        $pdo->prepare("
            UPDATE activations 
            SET last_check = NOW(), 
                site_name = :site_name,
                ip_address = :ip
            WHERE id = :id
        ")->execute(array(
                    ':id' => $existing_activation['id'],
                    ':site_name' => $site_name,
                    ':ip' => $_SERVER['REMOTE_ADDR']
                ));

    } elseif ($activation_count >= $license['max_sites']) {
        // تجاوز الحد الأقصى للمواقع
        send_json_response(array(
            'success' => false,
            'message' => 'تم الوصول للحد الأقصى من المواقع (' . $license['max_sites'] . '). قم بإلغاء تفعيل موقع آخر أولاً.'
        ), 403);
    } else {
        // تفعيل جديد
        $pdo->prepare("
            INSERT INTO activations 
            (license_id, site_url, site_name, ip_address, activated_at, last_check)
            VALUES 
            (:license_id, :site_url, :site_name, :ip, NOW(), NOW())
        ")->execute(array(
                    ':license_id' => $license['id'],
                    ':site_url' => $site_url,
                    ':site_name' => $site_name,
                    ':ip' => $_SERVER['REMOTE_ADDR']
                ));
    }

    // إرسال تنبيه تليجرام
    $log_msg = "🚀 <b>تفعيل جديد للبلجن!</b>\n";
    $log_msg .= "📦 المنتج: " . $product . "\n";
    $log_msg .= "🔑 الترخيص: <code>" . $license_key . "</code>\n";
    $log_msg .= "🌐 الموقع: " . $site_url . "\n";
    $log_msg .= "📧 مدير الموقع: " . $admin_email;
    send_telegram_alert($log_msg);

    // الاستجابة
    send_json_response(array(
        'success' => true,
        'message' => 'تم تفعيل الترخيص بنجاح!',
        'data' => array(
            'email' => $license_key, // تم تغييرها من email لـ license_key للتوافق
            'plan' => $license['plan'],
            'expires_at' => $license['expires_at'],
            'site_url' => $site_url,
            'max_sites' => $license['max_sites'],
            'activated_sites' => $activation_count + ($existing_activation ? 0 : 1)
        )
    ));

} catch (Exception $e) {
    send_json_response(array(
        'success' => false,
        'message' => 'Server error: ' . $e->getMessage()
    ), 500);
}

function sanitize_text_field($str)
{
    return htmlspecialchars(strip_tags(trim($str)), ENT_QUOTES, 'UTF-8');
}
